<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\DB;

use App\Models\Product;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use App\Models\ProductVariant;
use Illuminate\Support\Carbon;
use App\Services\BunnyCDNService;

class ProductController extends Controller
{
    protected $bunnyCDNService;

    public function __construct(BunnyCDNService $bunnyCDNService)
    {
        $this->bunnyCDNService = $bunnyCDNService;
    }
    public function search(Request $request)
    {
        $query = Product::with(['category', 'subcategory', 'brand', 'variants']);

        if ($request->has('search')) {
            $search = $request->input('search');
            $query->where('name', 'LIKE', "%{$search}%")
                ->orWhere('reference', 'LIKE', "%{$search}%")
                ->orWhereHas('category', function ($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%");
                })
                ->orWhereHas('subcategory', function ($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%");
                })
                ->orWhereHas('brand', function ($q) use ($search) {
                    $q->where('name', 'LIKE', "%{$search}%");
                });
        }

        if ($request->has('is_featured')) {
            $query->where('is_featured', $request->input('is_featured'));
        }
        if ($request->has('is_active')) {
            $query->where('is_active', $request->input('is_active'));
        }

        return response()->json($query->paginate(10));
    }


    public function create(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'required|exists:categories,id',
            'subcategory_id' => 'nullable|exists:sub_categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'reference' => 'required|string|max:50|unique:products,reference',
            'type' => 'required|in:demander_devis,display_price',
            'price' => 'required|numeric',
            'compare_price' => 'nullable|numeric',
            'stock' => 'required|integer|min:0',
            'sku' => 'nullable|string|max:50|unique:products,sku',
            'details' => 'nullable',
            'images' => 'nullable|array', // Array of images (Base64)
            'is_top_seller' => 'boolean',
            'is_special_offer' => 'boolean',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'variants' => 'nullable|array',
            'variants.*.name' => 'required|string|max:255',
            'variants.*.price' => 'nullable|numeric',
            'variants.*.stock' => 'required|integer|min:0',
            'variants.*.sku' => 'nullable|string|max:50|unique:product_variants,sku',
        ]);


        $uploadedImageUrls = [];

        if ($request->has('images')) {
            foreach ($request->input('images') as $base64Image) {
                $fileName = Str::uuid() . '.jpg';
                $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $base64Image));

                $tempDirectory = storage_path('app/public/temp');
                $tempFilePath = $tempDirectory . '/' . $fileName;

                if (!file_exists($tempDirectory)) {
                    mkdir($tempDirectory, 0777, true);
                }

                file_put_contents($tempFilePath, $imageData);

                $uploadedImageUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);
                $uploadedImageUrls[] = $uploadedImageUrl;

                unlink($tempFilePath);
            }
        }

        $productData = $validated;
        $productData['images'] = $uploadedImageUrls;

        $product = Product::create($productData);

        if ($request->has('variants')) {
            foreach ($request->input('variants') as $variant) {
                $product->variants()->create($variant);
            }
        }

        return response()->json([
            'message' => 'Product created successfully',
            'product' => $product
        ], 201);
    }




    public function get($id)
    {
        $product = Product::with(['category', 'subcategory', 'brand', 'variants'])->findOrFail($id);
        return response()->json($product);
    }


    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'category_id' => 'sometimes|required|exists:categories,id',
            'subcategory_id' => 'nullable|exists:sub_categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'reference' => 'sometimes|required|string|max:50|unique:products,reference,' . $product->id,
            'type' => 'sometimes|required|in:demander_devis,display_price',
            'price' => 'nullable|numeric',
            'compare_price' => 'nullable|numeric',
            'stock' => 'sometimes|required|integer|min:0',
            'sku' => 'nullable|string|max:50|unique:products,sku,' . $product->id,
            'details' => 'nullable',
            'images' => 'nullable|array',
            'is_top_seller' => 'boolean',
            'is_special_offer' => 'boolean',
            'is_featured' => 'boolean',
            'is_active' => 'boolean',
            'variants' => 'nullable|array',
            'variants.*.name' => 'required|string|max:255',
            'variants.*.price' => 'nullable|numeric',
            'variants.*.stock' => 'required|integer|min:0',
        ]);



        // Start with existing images
        $uploadedImageUrls =  [];

        if ($request->has('images')) {
            $newImageUrls = [];

            foreach ($request->input('images') as $image) {
                if (filter_var($image, FILTER_VALIDATE_URL)) {
                    // If it's a URL, add it directly
                    $newImageUrls[] = $image;
                } else {
                    // If it's base64, upload to BunnyCDN
                    $fileName = Str::uuid() . '.jpg';
                    $imageData = base64_decode(preg_replace('#^data:image/\w+;base64,#i', '', $image));

                    if ($imageData === false) {
                        return response()->json(['error' => 'Invalid image data'], 400);
                    }

                    $tempDirectory = storage_path('app/public/temp');
                    $tempFilePath = $tempDirectory . '/' . $fileName;

                    if (!file_exists($tempDirectory)) {
                        mkdir($tempDirectory, 0777, true);
                    }

                    file_put_contents($tempFilePath, $imageData);

                    $cdnUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);

                    if ($cdnUrl) {
                        $newImageUrls[] = $cdnUrl;
                    } else {
                        return response()->json(['error' => 'Image upload failed'], 500);
                    }

                    unlink($tempFilePath);
                }
            }

            // Merge existing and new images
            $uploadedImageUrls = array_merge($uploadedImageUrls, $newImageUrls);
        }

        // Update the product with the merged images
        $product->update(array_merge($validated, ['images' => $uploadedImageUrls]));

        // Handle variants
        if ($request->has('variants')) {
            foreach ($request->input('variants') as $variant) {
                $variant['updated_at'] = Carbon::now()->format('Y-m-d H:i:s');
                if (isset($variant['id'])) {
                    ProductVariant::where('id', $variant['id'])->update($variant);
                } else {
                    $product->variants()->create($variant);
                }
            }
        }

        return response()->json(['message' => 'Product updated successfully', 'product' => $product]);
    }





    public function delete($id)
    {
        $product = Product::findOrFail($id);
        $product->variants()->delete();
        $product->delete();

        return response()->json(['message' => 'Product deleted successfully']);
    }


    public function searchClientFeatured()
    {
        try {
            $products = Product::select(
                'products.id',
                'products.name',
                'products.description',
                'products.compare_price',
                'products.price',
                'products.details',
                'products.images',
                'products.is_special_offer',
                'products.is_featured',
                'products.is_top_seller',
                'products.created_at',
                DB::raw('COALESCE(AVG(reviews.rating), 0) as average_rating'),
                DB::raw('COUNT(reviews.id) as total_reviews')
            )
                ->leftJoin('reviews', 'reviews.product_id', '=', 'products.id')
                ->where('products.is_active', true)
                ->whereNull('products.deleted_at')
                ->where(function ($q) {
                    $q->where('products.is_top_seller', true)
                        ->orWhere('products.is_special_offer', true)
                        ->orWhere('products.is_featured', true);
                })
                ->groupBy(
                    'products.id',
                    'products.name',
                    'products.description',
                    'products.compare_price',
                    'products.price',
                    'products.details',
                    'products.images',
                    'products.is_special_offer',
                    'products.is_featured',
                    'products.is_top_seller',
                    'products.created_at'
                )
                ->orderBy('products.created_at', 'desc')
                ->get();

            return response()->json([
                'message' => 'Products retrieved successfully.',
                'data' => $products,
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An error occurred while retrieving products.',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }


    public function searchClient(Request $request)
    {
        try {
            $categories = $request->input('categories', []);
            $subcategories = $request->input('subcategories', []);
            $brands = $request->input('brands', []);
            $search = $request->input('search', '');
            $sort = $request->input('sort', '');
            $perPage = $request->input('per_page', 10);
            $page = $request->input('page', 1);

            $query = Product::select(
                'products.id',
                'products.name',
                'products.description',
                'products.compare_price',
                'products.price',
                'products.details',
                'products.images',
                'products.is_special_offer',
                'products.is_featured',
                'products.is_top_seller',
                'products.created_at',
                DB::raw('COALESCE(AVG(reviews.rating), 0) as average_rating'),
                DB::raw('COUNT(reviews.id) as total_reviews')
            )
                ->leftJoin('reviews', 'reviews.product_id', '=', 'products.id')
                ->where('products.is_active', true)
                ->whereNull('products.deleted_at')
                ->groupBy(
                    'products.id',
                    'products.name',
                    'products.description',
                    'products.compare_price',
                    'products.price',
                    'products.details',
                    'products.images',
                    'products.is_special_offer',
                    'products.is_featured',
                    'products.is_top_seller',
                    'products.created_at'
                );

            if (!empty($categories)) {
                $query->whereIn('products.category_id', $categories);
            }

            if (!empty($subcategories)) {
                $query->whereIn('products.subcategory_id', $subcategories);
            }

            if (!empty($brands)) {
                $query->whereIn('products.brand_id', $brands);
            }

            if (!empty($search)) {
                $query->where(function ($q) use ($search) {
                    $q->where('products.name', 'LIKE', "%$search%")
                        ->orWhere('products.description', 'LIKE', "%$search%");
                });
            }

            if (!empty($sort)) {
                if ($sort === 'plus récent') {
                    $query->orderBy('products.created_at', 'desc');
                } elseif ($sort === 'plus ancien') {
                    $query->orderBy('products.created_at', 'asc');
                } elseif ($sort === 'A-Z') {
                    $query->orderBy('products.name', 'asc');
                } elseif ($sort === 'Z-A') {
                    $query->orderBy('products.name', 'desc');
                } elseif ($sort === 'meilleure note') {
                    $query->orderBy('average_rating', 'desc'); // Sort by highest average rating
                }
            } else {
                $query->orderBy('products.created_at', 'desc');
            }

            $products = $query->paginate($perPage, ['*'], 'page', $page);

            return response()->json([
                'message' => 'Products retrieved successfully.',
                'data' => $products->items(),
                'current_page' => $products->currentPage(),
                'last_page' => $products->lastPage(),
                'total' => $products->total(),
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An error occurred while retrieving products.',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }




    public function getClient($id)
    {
        try {
            // Fetch the main product with reviews
            $product = Product::select(
                'products.id',
                'products.name',
                'description',
                'details',
                'compare_price',
                'price',
                'fiche_technique',
                'products.images',
                'is_special_offer',
                'is_featured',
                'is_top_seller',
                'products.created_at',
                'category_id',
                'subcategory_id',
                'brand_id',
                DB::raw('COALESCE(AVG(reviews.rating), 0) as average_rating'),
                DB::raw('COUNT(reviews.id) as total_reviews')
            )
                ->leftJoin('reviews', 'reviews.product_id', '=', 'products.id')
                ->where('products.id', $id)  // Explicitly refer to the 'products' table for the 'id' condition
                ->where('products.is_active', true)
                ->whereNull('products.deleted_at')
                ->groupBy(
                    'products.id',
                    'products.name',
                    'description',
                    'details',
                    'compare_price',
                    'price',
                    'fiche_technique',
                    'products.images',
                    'is_special_offer',
                    'is_featured',
                    'is_top_seller',
                    'products.created_at',
                    'category_id',
                    'subcategory_id',
                    'brand_id'
                )  // Group by all non-aggregated columns
                ->with('reviews', 'category', 'subcategory', 'brand')
                ->first();

            if (!$product) {
                return response()->json(['message' => 'No product found.'], Response::HTTP_NOT_FOUND);
            }

            // Fetch related products with reviews
            $relatedProducts = Product::select(
                'products.id',
                'products.name',
                'description',
                'compare_price',

                'products.images',
                'price',
                'is_special_offer',
                'is_featured',
                'is_top_seller',
                'products.created_at',
                DB::raw('COALESCE(AVG(reviews.rating), 0) as average_rating'),
                DB::raw('COUNT(reviews.id) as total_reviews')
            )
                ->leftJoin('reviews', 'reviews.product_id', '=', 'products.id')
                ->where('products.id', '!=', $id)
                ->where('products.is_active', true)
                ->whereNull('products.deleted_at')
                ->where(function ($query) use ($product) {
                    if ($product->category_id) {
                        $query->where('products.category_id', $product->category_id);
                    }
                    if ($product->subcategory_id) {
                        $query->where('products.subcategory_id', $product->subcategory_id);
                    }
                    if ($product->brand_id) {
                        $query->where('products.brand_id', $product->brand_id);
                    }
                })
                ->groupBy(
                    'products.id',
                    'products.name',
                    'description',
                    'products.images',
                    'price',
                    'compare_price',

                    'is_special_offer',
                    'is_featured',
                    'is_top_seller',
                    'products.created_at'
                )  // Group by all non-aggregated columns
                ->orderBy('products.created_at', 'desc')
                ->get();

            // Fallback: Get the newest 10 products if no related products
            if ($relatedProducts->isEmpty()) {
                $relatedProducts = Product::select(
                    'products.id',
                    'products.name',
                    'description',
                    'products.images',
                    'price',
                                        'compare_price',

                    'is_special_offer',
                    'is_featured',
                    'is_top_seller',
                    'products.created_at',
                    DB::raw('COALESCE(AVG(reviews.rating), 0) as average_rating'),
                    DB::raw('COUNT(reviews.id) as total_reviews')
                )
                    ->leftJoin('reviews', 'reviews.product_id', '=', 'products.id')
                    ->where('products.id', '!=', $id)
                    ->where('products.is_active', true)
                    ->whereNull('products.deleted_at')
                    ->groupBy(
                        'products.id',
                        'products.name',
                        'description',
                        'products.images',
                        'price',
                                            'compare_price',

                        'is_special_offer',
                        'is_featured',
                        'is_top_seller',
                        'products.created_at'
                    )  // Group by all non-aggregated columns
                    ->latest()
                    ->limit(10)
                    ->get();
            }

            return response()->json([
                'message' => 'Product retrieved successfully.',
                'data' => [
                    'product' => $product,
                    'related_products' => $relatedProducts
                ]
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An error occurred while retrieving product details.',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }
}
