<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Http\Response;
use App\Services\BunnyCDNService;
use Illuminate\Support\Facades\Validator;

class BrandController extends Controller
{
    protected $bunnyCDNService;

    public function __construct(BunnyCDNService $bunnyCDNService)
    {
        $this->bunnyCDNService = $bunnyCDNService;
    }
    private function isBase64Image(?string $value): bool
    {
        if (!$value || !is_string($value)) return false;
        return str_starts_with($value, 'data:image/');
    }

    private function uploadBase64ToBunny(?string $base64, string $ext = 'jpg'): ?string
    {
        if (!$this->isBase64Image($base64)) return $base64; // already URL or null

        $fileName = \Illuminate\Support\Str::uuid() . '.' . $ext;

        $imageData = base64_decode(
            preg_replace('#^data:image/\w+;base64,#i', '', $base64)
        );

        if (!$imageData) return null;

        $tempDirectory = storage_path('app/public/temp');
        if (!file_exists($tempDirectory)) {
            mkdir($tempDirectory, 0777, true);
        }

        $tempFilePath = $tempDirectory . '/' . $fileName;
        file_put_contents($tempFilePath, $imageData);

        $uploadedUrl = $this->bunnyCDNService->uploadImage($tempFilePath, $fileName);

        @unlink($tempFilePath);

        return $uploadedUrl ?: null;
    }

    /**
     * Accepts array of strings that can be:
     * - base64 images
     * - already hosted URLs
     * Returns array of URLs.
     */
    private function uploadManyBase64ToBunny($items): array
    {
        if (!is_array($items)) return [];

        $out = [];
        foreach ($items as $item) {
            if (is_string($item) && $this->isBase64Image($item)) {
                $url = $this->uploadBase64ToBunny($item);
                if ($url) $out[] = $url;
            } elseif (is_string($item) && trim($item) !== '') {
                // already a URL
                $out[] = $item;
            }
        }
        return $out;
    }

    public function createBrand(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'slug' => 'required|string|max:150|unique:brands,slug',
            'image' => 'nullable|string', // base64 or url
            'featured' => 'nullable|boolean',

            // SEO
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',

            // Hero slides (array of base64/url strings)
            'hero_slides' => 'nullable|array',
            'hero_slides.*' => 'nullable|string',

            // About section
            'about_title' => 'nullable|string|max:255',
            'about_text' => 'nullable|string',
            'about_images' => 'nullable|array',
            'about_images.*' => 'nullable|string',

            // Stats
            'about_stats' => 'nullable|array',
            'about_stats.*.value' => 'nullable|string|max:50',
            'about_stats.*.label' => 'nullable|string|max:100',

            // Accent
            'accent_color' => ['nullable', 'regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        // upload single brand image (if base64)
        $brandImageUrl = null;
        if ($request->filled('image')) {
            $brandImageUrl = $this->uploadBase64ToBunny($request->input('image'));
        }

        // upload hero slides -> store as [{id,image}]
        $heroSlidesInput = $request->input('hero_slides', []);
        $heroSlideUrls = $this->uploadManyBase64ToBunny($heroSlidesInput);

        $heroSlides = [];
        foreach ($heroSlideUrls as $i => $url) {
            $heroSlides[] = ['id' => $i + 1, 'image' => $url];
        }

        // upload about images (3 recommended, but allow any)
        $aboutImagesInput = $request->input('about_images', []);
        $aboutImages = $this->uploadManyBase64ToBunny($aboutImagesInput);

        $brand = Brand::create([
            'name' => $request->name,
            'slug' => $request->slug,
            'description' => $request->description,
            'image' => $brandImageUrl,
            'featured' => (bool) $request->featured,

            'meta_title' => $request->meta_title,
            'meta_description' => $request->meta_description,

            'hero_slides' => $heroSlides ?: null,
            'about_title' => $request->about_title,
            'about_text' => $request->about_text,
            'about_images' => $aboutImages ?: null,
            'about_stats' => $request->about_stats ?: null,
            'accent_color' => $request->accent_color,
        ]);

        return response()->json([
            'message' => 'Brand created successfully.',
            'data' => $brand,
        ], Response::HTTP_CREATED);
    }

    public function updateBrand(Request $request, $id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json(['message' => 'Brand not found.'], Response::HTTP_NOT_FOUND);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'description' => 'nullable|string',
            'slug' => 'required|string|max:150|unique:brands,slug,' . $brand->id,
            'image' => 'nullable|string',
            'featured' => 'nullable|boolean',

            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',

            'hero_slides' => 'nullable|array',
            'hero_slides.*' => 'nullable|string',

            'about_title' => 'nullable|string|max:255',
            'about_text' => 'nullable|string',
            'about_images' => 'nullable|array',
            'about_images.*' => 'nullable|string',

            'about_stats' => 'nullable|array',
            'about_stats.*.value' => 'nullable|string|max:50',
            'about_stats.*.label' => 'nullable|string|max:100',

            'accent_color' => ['nullable', 'regex:/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/'],
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => 'Validation failed.',
                'errors' => $validator->errors(),
            ], Response::HTTP_BAD_REQUEST);
        }

        // Brand image
        $brandImage = $brand->image;
        if ($request->filled('image')) {
            $incoming = $request->input('image');

            // Upload only if base64
            if ($this->isBase64Image($incoming)) {
                $uploaded = $this->uploadBase64ToBunny($incoming);
                if ($uploaded) $brandImage = $uploaded;
            } else {
                // URL string
                $brandImage = $incoming;
            }
        }

        // Hero slides
        $heroSlidesInput = $request->input('hero_slides', null);

        $heroSlides = $brand->hero_slides; // keep existing by default
        if (is_array($heroSlidesInput)) {
            $urls = $this->uploadManyBase64ToBunny($heroSlidesInput);
            $heroSlides = [];
            foreach ($urls as $i => $url) {
                $heroSlides[] = ['id' => $i + 1, 'image' => $url];
            }
        }

        // About images
        $aboutImagesInput = $request->input('about_images', null);
        $aboutImages = $brand->about_images;
        if (is_array($aboutImagesInput)) {
            $aboutImages = $this->uploadManyBase64ToBunny($aboutImagesInput);
        }

        // Save fields
        $brand->name = $request->name;
        $brand->slug = $request->slug;
        $brand->description = $request->description;
        $brand->image = $brandImage;
        $brand->featured = (bool) $request->featured;

        $brand->meta_title = $request->meta_title;
        $brand->meta_description = $request->meta_description;

        $brand->hero_slides = $heroSlides ?: null;
        $brand->about_title = $request->about_title;
        $brand->about_text = $request->about_text;
        $brand->about_images = $aboutImages ?: null;
        $brand->about_stats = $request->about_stats ?: null;
        $brand->accent_color = $request->accent_color;

        $brand->save();

        return response()->json([
            'message' => 'Brand updated successfully.',
            'data' => $brand
        ], Response::HTTP_OK);
    }



    public function delete($id)
    {
        $brand = Brand::find($id);

        if (!$brand) {
            return response()->json(['message' => 'Brand not found.'], Response::HTTP_NOT_FOUND);
        }


        $brand->delete();

        return response()->json(['message' => 'Brand deleted successfully.'], Response::HTTP_OK);
    }


    public function search(Request $request)
    {
        try {
            $searchValue = $request->input('searchValue');
            $regex = strtolower($searchValue);

            $page = $request->input('page');
            $perPage = $request->input('per_page', 10);

            $query = Brand::whereNull('deleted_at')
                ->orderBy('created_at', 'desc');

            if ($searchValue) {
                $query->where(function ($query) use ($regex) {
                    $query->whereRaw('LOWER(name) LIKE ?', ["%{$regex}%"]);
                });
            }

            if ($page) {


                $brands = $query->paginate($perPage, ['*'], 'page', $page);

                return response()->json([
                    'message' => 'Brands retrieved successfully.',
                    'data' => $brands->items(),
                    'total' => $brands->total(),
                    'current_page' => $brands->currentPage(),
                    'last_page' => $brands->lastPage(),
                ], Response::HTTP_OK);
            } else {
                $brands = $query->get();

                return response()->json([
                    'message' => 'Brands retrieved successfully.',
                    'data' => $brands,

                ], Response::HTTP_OK);
            }
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Internal Server Error',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }




    public function generateUniqueSlug($name, $model)
    {
        $slug = Str::slug($name);

        $originalSlug = $slug;
        $count = 1;

        while ($model::where('slug', $slug)->exists()) {
            $slug = $originalSlug . '-' . $count;
            $count++;
        }

        return $slug;
    }

    public function importBrands(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xls,xlsx|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => $validator->errors()->first(),
            ], 400);
        }

        try {
            $data = Excel::toCollection(null, $request->file('file'))->first();

            $importedCount = 0;
            $skippedCount = 0;

            foreach ($data as $key => $row) {
                if ($key === 0) {
                    continue;
                }

                if (empty($row[1])) {
                    $skippedCount++;
                    continue;
                }

                $slug = $this->generateUniqueSlug($row[1], Brand::class);

                $description = $row[3] ?? 'No description provided';

                Brand::create([
                    'name' => $row[1],
                    'slug' => $slug,
                    'description' => $description,
                ]);

                $importedCount++;
            }

            return response()->json([
                'success' => true,
                'message' => 'Brands imported successfully!',
                'imported' => $importedCount,
                'skipped' => $skippedCount,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'There was an error processing the file: ' . $e->getMessage(),
            ], 500);
        }
    }

    public function searchClient(Request $request)
    {
        try {
            $featured = $request->input('featured');

            $query = Brand::query()
                ->whereNull('deleted_at')
                ->orderBy('created_at', 'desc');

            if (!is_null($featured)) {
                $query->where('featured', filter_var($featured, FILTER_VALIDATE_BOOLEAN));
            }

            $brands = $query->get();

            return response()->json([
                'message' => 'Brands retrieved successfully.',
                'data' => $brands,
            ], Response::HTTP_OK);
        } catch (\Exception $e) {
            // Return a JSON response for any server error
            return response()->json([
                'message' => 'An error occurred while retrieving categories.',
                'error' => $e->getMessage(),
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }


    // Add this method INSIDE BrandController (same controller)
// GET /client/brands/get/{slugOrId}

public function getOneClient($slugOrId)
{
    try {
        $brand = Brand::query()
            ->whereNull('deleted_at')
            ->where(function ($q) use ($slugOrId) {
                if (is_numeric($slugOrId)) {
                    $q->where('id', (int) $slugOrId);
                }
                $q->orWhere('slug', $slugOrId);
            })
            ->first();

        if (!$brand) {
            return response()->json([
                'message' => 'Brand not found.',
                'data' => null,
            ], Response::HTTP_NOT_FOUND);
        }

        // Optional: provide safe defaults so frontend never breaks
        $brand->hero_slides = $brand->hero_slides ?: [];
        $brand->about_images = $brand->about_images ?: [];
        $brand->about_stats = $brand->about_stats ?: [];
        $brand->accent_color = $brand->accent_color ?: "#C4D457";

        return response()->json([
            'message' => 'Brand retrieved successfully.',
            'data' => $brand,
        ], Response::HTTP_OK);
    } catch (\Exception $e) {
        return response()->json([
            'message' => 'Internal Server Error',
            'error' => $e->getMessage(),
        ], Response::HTTP_INTERNAL_SERVER_ERROR);
    }
}

}
