<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\StoreContentTypeRequest;
use App\Http\Requests\Admin\UpdateContentTypeRequest;
use App\Http\Resources\Admin\ContentTypeResource;
use App\Models\ContentType;
use App\Services\Schema\SchemaNormalizer;

class ContentTypeController extends Controller
{
    public function index()
    {
        return ContentTypeResource::collection(ContentType::latest()->paginate(20));
    }

    public function store(StoreContentTypeRequest $request, SchemaNormalizer $normalizer)
    {
        $schema = $normalizer->normalize($request->input('schema', ['fields' => []]));

        $type = ContentType::create([
            'name' => $request->name,
            'slug' => $request->slug,
            'is_single' => (bool)$request->is_single,
            'locales' => $request->locales,
            'default_locale' => $request->default_locale,
            'schema' => $schema,
        ]);

        return new ContentTypeResource($type);
    }

    public function show(ContentType $contentType)
    {
        return new ContentTypeResource($contentType);
    }

    public function update(UpdateContentTypeRequest $request, ContentType $contentType, SchemaNormalizer $normalizer)
    {
        $schema = $request->has('schema')
            ? $normalizer->normalize($request->input('schema', $contentType->schema))
            : $contentType->schema;

        $contentType->update([
            'name' => $request->input('name', $contentType->name),
            'slug' => $request->input('slug', $contentType->slug),
            'is_single' => $request->has('is_single') ? (bool)$request->is_single : $contentType->is_single,
            'locales' => $request->input('locales', $contentType->locales),
            'default_locale' => $request->input('default_locale', $contentType->default_locale),
            'schema' => $schema,
        ]);

        return new ContentTypeResource($contentType->refresh());
    }

    public function destroy(ContentType $contentType)
    {
        $contentType->delete();
        return response()->json(['message' => 'Content type deleted.']);
    }
}
