<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\StoreContentEntryRequest;
use App\Http\Requests\Admin\UpdateContentEntryRequest;
use App\Http\Resources\Admin\ContentEntryResource;
use App\Models\ContentEntry;
use App\Models\ContentType;
use App\Services\Content\ContentCache;
use App\Services\Schema\SchemaDefaults;
use App\Services\Schema\SchemaRulesBuilder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ContentEntryController extends Controller
{
    public function index(Request $request, string $slug)
    {
        $type = ContentType::where('slug', $slug)->firstOrFail();

        $q = ContentEntry::where('content_type_id', $type->id)->latest();

        return ContentEntryResource::collection($q->paginate(20));
    }

    public function store(
        StoreContentEntryRequest $request,
        string $slug,
        SchemaRulesBuilder $builder,
        SchemaDefaults $defaults
    ) {
        $type = ContentType::where('slug', $slug)->firstOrFail();

        $data = $request->input('data', []);
        $data = $defaults->applyDefaults($type->schema, $data);

        $rules = $builder->build($type->schema, [
            'locales' => $type->locales ?? [$type->default_locale ?? 'en'],
            'default_locale' => $type->default_locale ?? 'en',
        ]);

        Validator::make($data, $rules)->validate();

        if ($type->is_single) {
            $entry = ContentEntry::where('content_type_id', $type->id)->latest()->first();

            if ($entry) {
                $entry->update([
                    'data' => $data,
                    'updated_by' => auth()->id(),
                ]);
            } else {
                $entry = ContentEntry::create([
                    'content_type_id' => $type->id,
                    'data' => $data,
                    'created_by' => auth()->id(),
                    'updated_by' => auth()->id(),
                ]);
            }

            return new ContentEntryResource($entry->refresh());
        }

        $entry = ContentEntry::create([
            'content_type_id' => $type->id,
            'data' => $data,
            'created_by' => auth()->id(),
            'updated_by' => auth()->id(),
        ]);

        return new ContentEntryResource($entry);
    }

    public function update(
        UpdateContentEntryRequest $request,
        ContentEntry $entry,
        SchemaRulesBuilder $builder,
        SchemaDefaults $defaults
    ) {
        $type = $entry->type()->firstOrFail();

        $data = $request->input('data', []);
        $data = $defaults->applyDefaults($type->schema, $data);

        $rules = $builder->build($type->schema, [
            'locales' => $type->locales ?? [$type->default_locale ?? 'en'],
            'default_locale' => $type->default_locale ?? 'en',
        ]);

        Validator::make($data, $rules)->validate();

        $entry->update([
            'data' => $data,
            'updated_by' => auth()->id(),
        ]);

        // If the entry is published, invalidate public cache
        if ($entry->status === 'published') {
            app(ContentCache::class)->forgetAllForSlug($type->slug);
        }

        return new ContentEntryResource($entry->refresh());
    }

    public function destroy(ContentEntry $entry)
    {
        $slug = $entry->type->slug;

        $entry->delete();

        app(ContentCache::class)->forgetAllForSlug($slug);

        return response()->json(['message' => 'Entry deleted.']);
    }

    public function publish(ContentEntry $entry)
    {
        $entry->update([
            'status' => 'published',
            'published_at' => now(),
            'updated_by' => auth()->id(),
        ]);

        app(ContentCache::class)->forgetAllForSlug($entry->type->slug);

        return response()->json(['message' => 'Published.']);
    }

    public function unpublish(ContentEntry $entry)
    {
        $entry->update([
            'status' => 'draft',
            'published_at' => null,
            'updated_by' => auth()->id(),
        ]);

        app(ContentCache::class)->forgetAllForSlug($entry->type->slug);

        return response()->json(['message' => 'Unpublished.']);
    }
}
